import { AsyncLocalStorage } from 'async_hooks';
import { DeepPropertyType, RecursiveKeyOf } from '../types/recursive-key-of.type';
import { AnyIfNever, StringIfNever, TypeIfUndefined } from '../types/type-if-type.type';
import { ClsContextOptions, ClsStore } from './cls.options';
export declare class ClsService<S extends ClsStore = ClsStore> {
    private readonly als;
    constructor(als: AsyncLocalStorage<any>);
    /**
     * Set (or overrides) a value on the CLS context.
     * @param key the key
     * @param value the value to set
     */
    set<R = undefined, T extends RecursiveKeyOf<S> = any, P extends DeepPropertyType<S, T> = any>(key: StringIfNever<T> | keyof ClsStore, value: AnyIfNever<P>): void;
    /**
     * Set a value on the CLS context if it doesn't already exist
     * @param key the key
     * @param value the value to set
     * @returns `true` if value vas set, `false` if it existed before
     */
    setIfUndefined<R = undefined, T extends RecursiveKeyOf<S> = any, P extends DeepPropertyType<S, T> = any>(key: StringIfNever<T> | keyof ClsStore, value: AnyIfNever<P>): boolean;
    /**
     * Retrieve the whole CLS context
     * @returns the value stored under the key or undefined
     */
    get(): AnyIfNever<S>;
    /**
     * Retrieve a value from the CLS context by key.
     * @param key the key from which to retrieve the value, returns the whole context if ommited
     * @returns the value stored under the key or undefined
     */
    get<R = undefined, T extends RecursiveKeyOf<S> = any, P = DeepPropertyType<S, T>>(key?: StringIfNever<T> | keyof ClsStore): TypeIfUndefined<R, TypeIfUndefined<typeof key, S, AnyIfNever<P>>, R>;
    /**
     * Check if a key is in the CLS context
     * @param key the key to check
     * @returns true if the key is in the CLS context
     */
    has<T extends RecursiveKeyOf<S> = any>(key: StringIfNever<T> | keyof ClsStore): boolean;
    /**
     * Retrieve the request ID (a shorthand for `cls.get(CLS_ID)`)
     * @returns the request ID or undefined
     */
    getId(): string;
    /**
     * Run the callback with a shared CLS context.
     * @returns whatever the callback returns
     */
    run<T = any>(callback: () => T): T;
    run<T = any>(options: ClsContextOptions, callback: () => T): T;
    /**
     * Run the callbacks with a new CLS context.
     *
     * @param store the default context contents
     * @param callback function to run
     * @returns whatever the callback returns
     */
    runWith<T = any>(store: S, callback: () => T): T;
    /**
     * Run any following code with a shared CLS context.
     */
    enter(): void;
    enter(options: ClsContextOptions): void;
    /**
     * Run any following code with a shared ClS context
     * @param store the default context contents
     */
    enterWith(store?: S): void;
    /**
     * Run the callback outside of a shared CLS context
     * @param callback function to run
     * @returns whatever the callback returns
     */
    exit<T = any>(callback: () => T): T;
    /**
     * Whether the current code runs within an active CLS context.
     * @returns true if a CLS context is active
     */
    isActive(): boolean;
    /**
     * Retrieve a Proxy provider from the CLS context
     * based on its injection token.
     */
    getProxy<T = any>(proxyToken: string | symbol): T;
    getProxy<T>(proxyToken: new (...args: any) => T): T;
    /**
     * Replace an instance of a Proxy provider in the CLS context
     * based on its injection token.
     */
    setProxy<T = any>(proxyToken: string | symbol, value: T): void;
    setProxy<T>(proxyToken: new (...args: any) => T, value: T): void;
    /**
     * Use to manually trigger resolution of Proxy Providers
     * in case `resolveProxyProviders` is not enabled in the enhancer.
     *
     * @param proxyTokens An optional array of Proxy Provider injection tokens
     * to resolve. If not supplied, resolves all registered proxy providers.
     */
    resolveProxyProviders(proxyTokens?: any[]): Promise<void>;
    /**
     * @deprecated This method will be removed in a future release and replaced
     * with a different mechanism for plugin initialization.
     *
     * Since the plugin API is still experimental, this method will become a np-op
     * and will be eventually removed, possibly in a minor release.
     */
    initializePlugins(): Promise<void>;
}
//# sourceMappingURL=cls.service.d.ts.map