"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.WorkflowService = void 0;
const common_1 = require("@nestjs/common");
const workflow_dto_1 = require("../dtos/workflow.dto");
const enum_1 = require("../enum");
const plugins_1 = require("../plugins");
const base_service_1 = require("./base.service");
let WorkflowService = class WorkflowService extends base_service_1.BaseService {
    async create(auth, dto) {
        const trigger = this.getTriggerOrFail(dto.triggerType);
        const filterInserts = await this.validateAndMapFilters(dto.filters, trigger.context);
        const actionInserts = await this.validateAndMapActions(dto.actions, trigger.context);
        const workflow = await this.workflowRepository.createWorkflow({
            ownerId: auth.user.id,
            triggerType: dto.triggerType,
            name: dto.name,
            description: dto.description || '',
            enabled: dto.enabled ?? true,
        }, filterInserts, actionInserts);
        return this.mapWorkflow(workflow);
    }
    async getAll(auth) {
        const workflows = await this.workflowRepository.getWorkflowsByOwner(auth.user.id);
        return Promise.all(workflows.map((workflow) => this.mapWorkflow(workflow)));
    }
    async get(auth, id) {
        await this.requireAccess({ auth, permission: enum_1.Permission.WorkflowRead, ids: [id] });
        const workflow = await this.findOrFail(id);
        return this.mapWorkflow(workflow);
    }
    async update(auth, id, dto) {
        await this.requireAccess({ auth, permission: enum_1.Permission.WorkflowUpdate, ids: [id] });
        if (Object.values(dto).filter((prop) => prop !== undefined).length === 0) {
            throw new common_1.BadRequestException('No fields to update');
        }
        const workflow = await this.findOrFail(id);
        const trigger = this.getTriggerOrFail(workflow.triggerType);
        const { filters, actions, ...workflowUpdate } = dto;
        const filterInserts = filters && (await this.validateAndMapFilters(filters, trigger.context));
        const actionInserts = actions && (await this.validateAndMapActions(actions, trigger.context));
        const updatedWorkflow = await this.workflowRepository.updateWorkflow(id, workflowUpdate, filterInserts, actionInserts);
        return this.mapWorkflow(updatedWorkflow);
    }
    async delete(auth, id) {
        await this.requireAccess({ auth, permission: enum_1.Permission.WorkflowDelete, ids: [id] });
        await this.workflowRepository.deleteWorkflow(id);
    }
    async validateAndMapFilters(filters, requiredContext) {
        for (const dto of filters) {
            const filter = await this.pluginRepository.getFilter(dto.pluginFilterId);
            if (!filter) {
                throw new common_1.BadRequestException(`Invalid filter ID: ${dto.pluginFilterId}`);
            }
            if (!filter.supportedContexts.includes(requiredContext)) {
                throw new common_1.BadRequestException(`Filter "${filter.title}" does not support ${requiredContext} context. Supported contexts: ${filter.supportedContexts.join(', ')}`);
            }
        }
        return filters.map((dto, index) => ({
            pluginFilterId: dto.pluginFilterId,
            filterConfig: dto.filterConfig || null,
            order: index,
        }));
    }
    async validateAndMapActions(actions, requiredContext) {
        for (const dto of actions) {
            const action = await this.pluginRepository.getAction(dto.pluginActionId);
            if (!action) {
                throw new common_1.BadRequestException(`Invalid action ID: ${dto.pluginActionId}`);
            }
            if (!action.supportedContexts.includes(requiredContext)) {
                throw new common_1.BadRequestException(`Action "${action.title}" does not support ${requiredContext} context. Supported contexts: ${action.supportedContexts.join(', ')}`);
            }
        }
        return actions.map((dto, index) => ({
            pluginActionId: dto.pluginActionId,
            actionConfig: dto.actionConfig || null,
            order: index,
        }));
    }
    getTriggerOrFail(triggerType) {
        const trigger = plugins_1.pluginTriggers.find((t) => t.type === triggerType);
        if (!trigger) {
            throw new common_1.BadRequestException(`Invalid trigger type: ${triggerType}`);
        }
        return trigger;
    }
    async findOrFail(id) {
        const workflow = await this.workflowRepository.getWorkflow(id);
        if (!workflow) {
            throw new common_1.BadRequestException('Workflow not found');
        }
        return workflow;
    }
    async mapWorkflow(workflow) {
        const filters = await this.workflowRepository.getFilters(workflow.id);
        const actions = await this.workflowRepository.getActions(workflow.id);
        return {
            id: workflow.id,
            ownerId: workflow.ownerId,
            triggerType: workflow.triggerType,
            name: workflow.name,
            description: workflow.description,
            createdAt: workflow.createdAt.toISOString(),
            enabled: workflow.enabled,
            filters: filters.map((f) => (0, workflow_dto_1.mapWorkflowFilter)(f)),
            actions: actions.map((a) => (0, workflow_dto_1.mapWorkflowAction)(a)),
        };
    }
};
exports.WorkflowService = WorkflowService;
exports.WorkflowService = WorkflowService = __decorate([
    (0, common_1.Injectable)()
], WorkflowService);
//# sourceMappingURL=workflow.service.js.map